// Solace -- Sol Anachronistic Computer Emulation
// A Win32 emulator for the Sol-20 computer.
//
// Copyright (c) Jim Battle, 2002

#include <stdio.h>
#include <stdlib.h>
#include <io.h>		// required for open_osfhandle() function
#include <fcntl.h>	// file open/close
#include <sys/stat.h>	// needed for file permission flags

#include <windows.h>	// main windows includes
#include <windowsx.h>	// message cracker macros
#include <mmsystem.h>	// needed for waveOut stuff

#include "solace_intf.h" // core emulator definitions
#include "wingui.h"	// GUI-specific definitions
#include "wav.h"	// wav file description

#define K *1024
#define WAVE_BUF_SAMPLES (2 K)	// # of samples per buffer (free variable)
#define NUM_WAVE_BUFFERS  6	// # of audio buffers      (free variable)
#define BUFFER_LOW_THOLD  2	// # of full audio buffers before throttling 8080
#define BUFFER_HIGH_THOLD 4	// # of full audio buffers before throttling 8080

// ========================================================================
//  module globals
// ========================================================================

static WAVEHDR wavehdr[NUM_WAVE_BUFFERS];

static HWAVEOUT hAudio;		// handle to open audio channel
static int audio_opened;	// is the audio out channel open?
static int sample_bits;		// # bits per sample
static int sample_bytes;	// # bytes per sample
static int sample_rate;		// # samples per second

static int buffs;		// # of buffers tied up in waveOut
static int bufptr;		// which buffer we're writing to
static int sampptr;		// sample # of buffer we're writing to

// forward declarations:
static int WinAudioAttempt(void);
static void throttle_cpu(int buffs);


// ---- stuff for waveform capture ----

#define LOG_LIMIT_MINUTES 15	// 44.1KHz, 16bpp = 5 MB/minute

#define OUTBUFSIZE 4096		// transfer block size, in byte

static int Outfile;			// handle to output file
static int out_putptr;			// output buffer put pointer
static int out_total_samples;		// # of samples in the file
static uint8 outbuf[OUTBUFSIZE*2];	// holds items (may be 8b or 16b)

static void SaveWaveSample(float v);
static void SaveWaveFlush(void);


// ========================================================================
//   start of code
// ========================================================================

// called once at start of emulation
void
WinAudioInit(void)
{
    int i;

    // create N buffers for multi-buffering audio
    for(i=0; i<NUM_WAVE_BUFFERS; i++)
	wavehdr[i].lpData = (LPSTR)0;

    // module global variable initialization
    hAudio       = (HWAVEOUT)0;
    audio_opened = 0;	// no channel
    WinAudioSet(WA_PROP_SAMPLERATE,    44100);
    WinAudioSet(WA_PROP_BITSPERSAMPLE, 16);

    Sys_ThrottleScheduler(Throttle_NORMAL);

    Outfile = -1;	// handle to output file
}


// set properties on the output stream.
// can be used only if the channel is closed.
int
WinAudioSet(int prop, int value)
{
    // can't change these if audio is active
    if (audio_opened)
	return WA_STAT_OK;

    switch (prop) {

	case WA_PROP_SAMPLERATE:
	    switch (value) {
		case 44100:	// sounds like crap
		case 22050:	// sounds like crap
		// case 11025:  // sounds like crap^2
		    break;
		default:
		    return WA_STAT_BADVALUE;
	    }
	    sample_rate = value;
	    break;

	case WA_PROP_BITSPERSAMPLE:
	    switch (value) {
		case  8:
		case 16:
		    break;
		default:
		    return WA_STAT_BADVALUE;
	    }
	    sample_bits  =  value;
	    sample_bytes = (sample_bits/8);
	    break;

	default:
	    return WA_STAT_BADPROP;
    }

    return WA_STAT_OK;
}


// get properties of the output stream.
int
WinAudioGet(int prop, int *value)
{
    switch (prop) {

	case WA_PROP_SAMPLERATE:
	    *value = sample_rate;
	    break;

	case WA_PROP_BITSPERSAMPLE:
	    *value = sample_bits;
	    break;

	case WA_PROP_CAPTURING:
	    *value = (Outfile >= 0);
	    break;

	default:
	    return WA_STAT_BADPROP;
    }

    return WA_STAT_OK;
}


static void CALLBACK
audio_waveOutProc(
	HWAVEOUT hwo,
	UINT uMsg,
	DWORD dwInstance,
	DWORD dwParam1,
	DWORD dwParam2)
{
    switch (uMsg) {
	case WOM_OPEN:
	    break;
	case WOM_CLOSE:
	    break;
	case WOM_DONE:
	    buffs--;
	    throttle_cpu(buffs);
	    break;
    }
}


// open an audio channel, trying high quality settings first
// then falling back to worse sample rates and bit depths.
// returns WA_STAT_OK if we got a channel, and WA_STAT_NOAUDIO otherwise.
int
WinAudioOpen(void)
{
    int stat;

    if (audio_opened)
	return WA_STAT_OK;

    stat = WinAudioSet(WA_PROP_SAMPLERATE, 44100);
    ASSERT(stat == WA_STAT_OK);
    stat = WinAudioSet(WA_PROP_BITSPERSAMPLE, 16);
    ASSERT(stat == WA_STAT_OK);
    if (WinAudioAttempt() == WA_STAT_OK) {
	sample_rate  = 44100;
	sample_bits  = 16;
	audio_opened = 1;
	return WA_STAT_OK;
    }

    stat = WinAudioSet(WA_PROP_SAMPLERATE, 22050);
    ASSERT(stat == WA_STAT_OK);
    stat = WinAudioSet(WA_PROP_BITSPERSAMPLE, 16);
    ASSERT(stat == WA_STAT_OK);
    if (WinAudioAttempt() == WA_STAT_OK) {
	audio_opened = 1;
	return WA_STAT_OK;
    }

    stat = WinAudioSet(WA_PROP_SAMPLERATE, 44100);
    ASSERT(stat == WA_STAT_OK);
    stat = WinAudioSet(WA_PROP_BITSPERSAMPLE, 8);
    ASSERT(stat == WA_STAT_OK);
    if (WinAudioAttempt() == WA_STAT_OK) {
	audio_opened = 1;
	return WA_STAT_OK;
    }

    stat = WinAudioSet(WA_PROP_SAMPLERATE, 22050);
    ASSERT(stat == WA_STAT_OK);
    stat = WinAudioSet(WA_PROP_BITSPERSAMPLE, 8);
    ASSERT(stat == WA_STAT_OK);
    if (WinAudioAttempt() == WA_STAT_OK) {
	audio_opened = 1;
	return WA_STAT_OK;
    }

    audio_opened = 0;
    UI_Alert("Can't find a suitable audio device that is available");
    return WA_STAT_NOAUDIO;
}


// try to open a channel at the specified rate
static int
WinAudioAttempt(void)
{
    MMRESULT rv;
    WAVEFORMATEX wfx;
    int sample_bytes = (sample_bits/8);
    int i, error;

    ASSERT(!audio_opened);

    // describe what type of playback format we want
    wfx.wFormatTag      = WAVE_FORMAT_PCM;	// raw
    wfx.nChannels       = 1;			// mono
    wfx.nSamplesPerSec  = sample_rate;
    wfx.wBitsPerSample  = sample_bits;
    wfx.nBlockAlign     = wfx.nChannels * sample_bytes;
    wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nBlockAlign;
    wfx.cbSize          = 0;

    rv = waveOutOpen( &hAudio, WAVE_MAPPER, &wfx, (DWORD)audio_waveOutProc,
		      0,		// dwCallbackInstance,
		      CALLBACK_FUNCTION	// or maybe CALLBACK_NULL
    );
#if 0
    switch (rv) {
	case MMSYSERR_NOERROR:     break;
	case MMSYSERR_ALLOCATED:   UI_Alert("Specified resource is already allocated."); break;
	case MMSYSERR_BADDEVICEID: UI_Alert("Specified device identifier is out of range."); break;
	case MMSYSERR_NODRIVER:    UI_Alert("No device driver is present."); break;
	case MMSYSERR_NOMEM:       UI_Alert("Unable to allocate or lock memory."); break;
	case WAVERR_BADFORMAT:     UI_Alert("Attempted to open with an unsupported waveform-audio format."); break;
	case WAVERR_SYNC:          UI_Alert("The device is synchronous but waveOutOpen was called without using the WAVE_ALLOWSYNC flag."); break;
	default:                   UI_Alert("Unknown error opening audio device"); break;
    }
#endif
    if (rv != MMSYSERR_NOERROR)
	return WA_STAT_NOAUDIO;


    // create N buffers for multi-buffering audio

    error = 0;
    for(i=0; i<NUM_WAVE_BUFFERS; i++) {

	int bufferlen = (sample_bytes * WAVE_BUF_SAMPLES);

	wavehdr[i].lpData          = (LPSTR)malloc(bufferlen);
	wavehdr[i].dwBufferLength  = (DWORD)(bufferlen);
	wavehdr[i].dwUser          = (DWORD)0;	// unused
	wavehdr[i].dwFlags         = (DWORD)0;	// status
	wavehdr[i].dwLoops         = (DWORD)1;	// unused?

	if (wavehdr[i].lpData == NULL) {
	    error = 1;
	} else {
	    rv = waveOutPrepareHeader(hAudio, &wavehdr[i], sizeof(WAVEHDR));
	    error = (rv != MMSYSERR_NOERROR);
	}

	if (error)
	    break;
    }

    if (error) {
	WinAudioClose();
    } else {
	audio_opened = 1;
	buffs        = 0;
	bufptr       = 0;
	sampptr      = 0;
    }

    // tell core emulator how often to proceed
    Sys_ThrottleScheduler(Throttle_NORMAL);
    Sys_SetAudioSamplerate(sample_rate);

    return (error) ? WA_STAT_NOAUDIO : WA_STAT_OK;
}


// close the audio channel if open and release any allocated resources
void
WinAudioClose(void)
{
    int i;

    // close the audio channel
    if (hAudio != (HWAVEOUT)0) {
	waveOutReset(hAudio);
	for(i=0; i<NUM_WAVE_BUFFERS; i++) {
	    waveOutUnprepareHeader(hAudio, &wavehdr[i], sizeof(WAVEHDR));
	    free(wavehdr[i].lpData);
	    wavehdr[i].lpData = NULL;
	}
	waveOutClose(hAudio);
	hAudio = (HWAVEOUT)0;
    }

    // let emu run at full speed in case we throttled it earlier
    Sys_ThrottleScheduler(Throttle_NORMAL);

    audio_opened = 0;
}


// the input varies between 0.0 and 1.0
void
UI_CreateAudioSample(float v)
{
    uint8 *buff8;
    int16 *buff16;
    MMRESULT rv;

    SaveWaveSample(v);	// save to log file if requested

    if (!audio_opened ||		// audio not enabled
	(buffs == NUM_WAVE_BUFFERS)) {	// no place to put the sample
	return;
    }

    switch (sample_bits) {
	case  8:
	    buff8  = (uint8*)(wavehdr[bufptr].lpData);
	    buff8[sampptr] = (uint8)(254.0f*(v) + 0.5f);
	    break;
	case 16:
	    buff16 = (int16*)(wavehdr[bufptr].lpData);
	    buff16[sampptr] = (int16)(65530.0f*((v)-0.5f) + 0.5f);
	    break;
	default:
	    ASSERT(0);
	    break;
    }

    sampptr++;
    if (sampptr >= WAVE_BUF_SAMPLES) {
	rv = waveOutWrite(hAudio, &wavehdr[bufptr], sizeof(WAVEHDR));
	buffs++;
	throttle_cpu(buffs);
	if (++bufptr >= NUM_WAVE_BUFFERS)
	    bufptr = 0;
	sampptr = 0;
    }
}

// if we are in danger of underrunning the waveout buffer,
// let the cpu run for more cycles per timeslice than normal.
// if we are in danger of overrunning the waveout buffer,
// cut back the number of cpu cycles per timeslice.
int thist[32];
int thistptr = 0;
static void
throttle_cpu(int buffs)
{
    ASSERT(buffs >= 0);
    ASSERT(buffs <= NUM_WAVE_BUFFERS);

thist[thistptr++] = buffs;
thistptr &= 31;

    if (buffs <= BUFFER_LOW_THOLD)
	Sys_ThrottleScheduler(Throttle_FASTER);
    else if (buffs >= BUFFER_HIGH_THOLD)
	Sys_ThrottleScheduler(Throttle_SLOWER);
    else
	Sys_ThrottleScheduler(Throttle_NORMAL);
}


// ========================================================================
//   log samples to a file
// ========================================================================

// write a header to the wave file.  it is called twice per file: once
// to just reserve the space, and once when the file is closed to write
// an accurate value for the # of samples in file (DataHdr.chunkSize).
static void
SaveWaveWriteHeader(uint32 samples)
{
    RIFF_t        RiffHdr;
    FormatChunk_t FormatHdr;
    DataChunk_t   DataHdr;
    int databytes = samples * sample_bytes;

    ASSERT(Outfile >= 0);

    RiffHdr.groupID   = RiffID;
    RiffHdr.riffBytes = sizeof(RIFF_t)-8
		      + sizeof(FormatChunk_t)
		      + sizeof(DataChunk_t)
		      + databytes;
    RiffHdr.riffType  = WaveID;

    FormatHdr.chunkID       = FmtID;
    FormatHdr.chunkSize     = sizeof(FormatHdr)-8;
    FormatHdr.FormatTag     = 1;		// uncompressed
    FormatHdr.Channels      = 1;		// mono
    FormatHdr.Frequency     = sample_rate;	// sample frequency
    FormatHdr.BitsPerSample = sample_bits;
    FormatHdr.BlockAlign    = FormatHdr.Channels*sample_bytes;
    FormatHdr.AvgBPS        = FormatHdr.BlockAlign*sample_rate;

    DataHdr.chunkID   = DataID;
    DataHdr.chunkSize = databytes;

    // now write it out
    if (write(Outfile, (char*)&RiffHdr, sizeof(RiffHdr)) != sizeof(RiffHdr)) {
	printf("Error writing to output file\n");
	exit(-1);
    }
    if (write(Outfile, (char*)&FormatHdr, sizeof(FormatHdr)) != sizeof(FormatHdr)) {
	printf("Error writing to output file\n");
	exit(-1);
    }
    if (write(Outfile, (char*)&DataHdr, sizeof(DataHdr)) != sizeof(DataHdr)) {
	printf("Error writing to output file\n");
	exit(-1);
    }
}


// this creates an output file for WAV data.  it is always mono, but
// the sample rate and sample size are determined by the current
// audio channel settings.
int
SaveWaveOpen(char *name)
{
    const PERMS = _S_IREAD | _S_IWRITE;

    ASSERT(Outfile < 0);

    Outfile = open(name, O_WRONLY | O_CREAT | O_BINARY, PERMS);
    if (Outfile < 0) {
	UI_Alert("Error: couldn't open output file '%s'\n", name);
	return 0;
    }

    SaveWaveWriteHeader(60*sample_rate*LOG_LIMIT_MINUTES);	// create file

    out_total_samples = 0;
    out_putptr = 0;

    return 1;
}


// this routine accepts samples and accumulates them into a buffer
// and then dumps the buffer when full.
//
// the total number of samples must be less than 2**32 or we die.

static void
SaveWaveSample(float v)
{
    uint8 *buff8  = (uint8*)outbuf;
    int16 *buff16 = (int16*)outbuf;

    if (Outfile < 0)
	return;	// file isn't opened

    switch (sample_bits) {
	case  8:
	    buff8[out_putptr]  = (uint8)(255.0f*(v) + 0.5f);
	    break;
	case 16:
	    buff16[out_putptr] = (int16)(65535.0f*((v)-0.5f) + 0.5f);
	    break;
	default:
	    ASSERT(0);
	    break;
    }

    out_putptr++;
    if (out_putptr >= OUTBUFSIZE)
	SaveWaveFlush();

    out_total_samples++;
    if (out_total_samples > (60*sample_rate*LOG_LIMIT_MINUTES))
	SaveWaveClose();
}


static void
SaveWaveFlush(void)
{
    int bytes = out_putptr * sample_bytes;

    if (out_putptr > 0) {

	out_putptr = 0;

	// write the data
	if (write(Outfile, outbuf, bytes) != bytes) {
	    SaveWaveClose();
	    UI_Alert("Error writing audio to file -- closing it\n");
	}
    }
}


void
SaveWaveClose(void)
{
    ASSERT(Outfile >= 0);

    // save any partial buffer
    SaveWaveFlush();

    // rewind to start of file and write accurate header
    lseek(Outfile, 0, 0);
    SaveWaveWriteHeader(out_total_samples);

    close(Outfile);
    Outfile = -1;
}

